import json
import os

import sys
from datetime import datetime
import time
import requests

VERSION = 8.2

def getDeviceConfig(**kwargs):

    retry = kwargs.get("retry", False)

    dir_path = os.path.dirname(os.path.realpath(__file__))
    configFile = os.path.join(dir_path, "BackhaulConfiguration.conf")

    try:
        with open(configFile, "r") as f:
            config = json.loads(f.read())
            config["ConfigurationPath"] = configFile
            return config
    except Exception as e:
        print("File possibly JSON Corrupted or Not Valid -- Check for misplaced commas" )
        if retry:
            pass
        print("Attempting to read from Backup...")   
        backupFile = os.path.join(dir_path, "BackhaulConfigurationBackup.conf")
        if os.path.exists(backupFile) and retry is not True:
            if readBackup():
                print("Reading from Backup Successful!")
                if readBackup():
                    time.sleep(2)
                    return getDeviceConfig(retry=True)
            else:
                print("Reading from Backup FAILED.")

    print("No Configuration File or Backup! Backhaul Cannot run without valid configuration!")
    exit()

def generateConfigurationFile():
    # Any new Variables put in here, along with their default values if able.
    # Ensures configuration files are up to date
    # Idk should we? WIP

    device = getDeviceConfig()
    configTuple = [("SendType", "Pulsar"), 
                        ("Version", VERSION), 
                        ("WebsocketTopics", []),
                        ("WebsocketURL", ""), 
                        ("BasicAuth", False),
                        ("wsUser", ""),
                        ("wsPassword", ""),
                        ("HeartbeatInterval", 600),
                        ("ServiceMonitorID", None)]

    for config in configTuple:
        if not device.has_key(config[0]):
            print(str(config[0]) + " not found. Saving to configuration file.")
            device[config[0]] = config[1]
        else:
            print(str(config[0] + " already exists!"))

    saveAllConfig(device)


def getDeviceType():
    device = getDeviceConfig()
    return device['Device']

def getSerial():

    device = getDeviceConfig()
    if device is not None:
        if device['Device'] == "Counter":
            sys.path.insert(0, '/var/www/cgi-bin')
            import serial
            return str(serial.getserialN())
        
        if device['Device'] == "RPI":
            sys.path.insert(0, '/var/www/cgi-bin')
            import serial
            return str(serial.getserialN())

        if device['Device'] == "Jetson":
            sys.path.insert(0, '/var/www/cgi-bin')
            import jetsonSerial
            return str(jetsonSerial.jetsonSerial())

        if device['Device'] == "MM":
            if os.path.exists("C:\\MagicMirror\\UnityHost\\UnityCore_Data\\Json\\Device.json"):
                with open("C:\\MagicMirror\\UnityHost\\UnityCore_Data\\Json\\Device.json") as file:
                    contents = file.read()
                    contents = json.loads(contents)
                    return str(contents['Id'])
        if device['Device'] in ["Server", "DebugMachine"]:
            return device['DeviceID']

    return None

def getConfigurationTopic():
    device = getDeviceConfig()
    if device is not None:
        if device['Device'] == "Counter":
            sys.path.insert(0, '/var/www/cgi-bin')
            import serial
            return "ffc-configuration-"+str(serial.getserialN())
        
        if device['Device'] == "RPI":
            sys.path.insert(0, '/var/www/cgi-bin')
            import serial
            return "rpi-configuration-"+str(serial.getserialN())

        if device['Device'] == "Jetson":
            sys.path.insert(0, '/var/www/cgi-bin')
            import jetsonSerial
            return "jetson-configuration-"+str(jetsonSerial.jetsonSerial())

        # if device['Device'] == "MM":
        #     if os.path.exists("C:\\MagicMirror\\UnityHost\\UnityCore_Data\\Json\\Device.json"):
        #         with open("C:\\MagicMirror\\UnityHost\\UnityCore_Data\\Json\\Device.json") as file:
        #             contents = file.read()
        #             contents = json.loads(contents)
        #             return str(contents['Id'])

        if device['Device'] in ["Server", "DebugMachine"]:
            print("No configuration topic for this device!")
            return "generic-test-topic"

    return None

def getPulsarAddress():
    device = getDeviceConfig()
    try:
        if device is not None:
            return device['PulsarAddress']
        return None
    except Exception as e:
        return None

def getServerAddress():
    # Get Server Address from Windows or Counter. 
    device = getDeviceConfig()
    if device["Device"] == "Counter":
        import sqlite3
        conn = sqlite3.connect("/home/pi/Raspicam/raspicam")
        c = conn.cursor()
        c.execute('select Server from camera;')
        address = c.fetchone()[0] + "/"
    elif device["Device"] == "RPI":
        import sqlite3
        conn = sqlite3.connect("/home/pi/Rpi/rpi")
        c = conn.cursor()
        c.execute('select Server from camera;')
        address = c.fetchone()[0] + "/"
    elif device["Device"] == "Jetson":
        import sqlite3
        conn = sqlite3.connect("/home/pi/Jetson/jetson")
        c = conn.cursor()
        c.execute('select Server from jetson;')
        address = c.fetchone()[0] + "/"
    elif device["Device"] == "MM":
        return "http://footfallcounter.com/"
    elif device["Device"] in ["Server", "DebugMachine"]:
        # Config.json contains the server address for Windows machines
        # Location is back two directories to find RetailCamModulesConfig
        configPath = os.path.abspath(os.path.join(__file__ ,"../../../RetailCamModulesConfig/Config.json"))
        print("Attempt to find Server Address in: " + configPath)
        if os.path.exists(configPath):
            with open(configPath) as file:
                contents = file.read()
                contents = json.loads(contents)
                address =  contents['RetailCamModuleURL']
        else:
            return None
    else:
        print("Invalid Device Type: " + str(device["Device"]))
        exit()

    return address

def updatePulsarAddress():

    try:
        device = getDeviceConfig()
        serverAddress = getServerAddress()
        if serverAddress is not None:
            if device["Device"] in ["Counter", "RPI", "Jetson"]:
                apiURL = "api/action/getsystemsettings?key=PulsarServerAddress&serial="+getSerial()
            else:
                apiURL = "api/action/getsystemsettings?key=PulsarServerAddress"
            url = serverAddress + apiURL
            request = requests.get(url, timeout=30)
            if request.status_code == 200:
                data = json.loads(request.text)
                pulsarAddress =  "ws://" + data["Data"]["Host"] + ":" + data["Data"]["Port"]
                saveConfig("PulsarAddress", pulsarAddress)
                saveConfig("Version", VERSION)
                return True
            else:
                print("FAILED to get Pulsar Server Address via API Call")
                return False
        else:
            print("Portal Server Address not found!")
            return False
    except Exception as e:
        print(e)
        return False

def getIDFormat():

    default = "DeviceId"
    device = getDeviceConfig()
    if device is not None:
        return default if device['DeviceIDFormat'] is None else device['DeviceIDFormat']
    return "DeviceId"

def getRMQServerPort():

    try:
        device = getDeviceConfig()
        if device is not None:
            return device['RMQServerPort']
        return None
    except Exception as e:
        print("RMQServerPort is not found or invalid. Using default port: 8088")
        return 8088

def getLogDirectory():

    device = getDeviceConfig()
    if device['LogDirectory'] is None:
        #Server, MM to generate a 
        dir_path = os.path.dirname(os.path.realpath(__file__))
        logPath = os.path.join(dir_path, "PulsarLogs")
        cleanLogDirectory()
        if not os.path.exists(logPath):
            os.mkdir(logPath)
        return logPath + "\\"
    else:
        return device['LogDirectory']

def cleanLogDirectory():
    dir_path = os.path.dirname(os.path.realpath(__file__))
    logPath = os.path.join(dir_path, "PulsarLogs")
    current_time = time.time()
    for f in os.listdir(logPath):
        fileName = os.path.join(dir_path, "PulsarLogs", f)
        creation_time = os.path.getctime(os.path.join(dir_path, "PulsarLogs", fileName))
        if (current_time - creation_time) // (24 * 3600) >= 7:
            if os.path.exists(fileName):
                try:
                    os.remove(fileName)
                    print('{} removed'.format(fileName))
                except Exception as e:
                    pass
    return True

def getLocalDateTimeFormat():
    device = getDeviceConfig()
    if device is not None:
        return "UploadedLocalDateTime" if device['UploadedLocalDateTimeFormat'] is None else device['UploadedLocalDateTimeFormat']
    return None

def getUTCDateTimeFormat():
    device = getDeviceConfig()
    if device is not None:
        return "UploadedUTCDateTime" if device['UploadedUTCDateTimeFormat'] is None else device['UploadedUTCDateTimeFormat']
    return None

def getUpdateInterval():
    try:
        device = getDeviceConfig()
        if device is not None:
            return device["UpdateInterval"]
    except KeyError as e:
        saveConfig("UpdateInterval", 1800)
        return 3600

def getRestartInterval():
    try:
        device = getDeviceConfig()
        if device is not None:
            return device["RestartInterval"]
    except KeyError as e:
        saveConfig("RestartInterval", 60)
        return 300

def saveConfig(config, value):
    try:
        device = getDeviceConfig()
        if device is not None:
            device[config] = value
            path = device.pop("ConfigurationPath")
            with open(path, 'w') as file:
                file.write(json.dumps(json.loads(json.dumps(device)), indent=4, sort_keys=True))
                print("{0} saved. New Value: {1}".format(config, value))
            return True
    except Exception as e:
        print("Unable to save Config: " + str(e))
        exit()

def saveAllConfig(config):
    device = getDeviceConfig()
    path = device.pop("ConfigurationPath")
    try:
        with open(path, 'w') as file:
            file.write(json.dumps(json.loads(json.dumps(config)), indent=4, sort_keys=True))
        return True
    except Exception as e:
        print("Unable to save Config: " + str(e))
        exit()
    return False

def deleteConfig(config):
    try:
        device = getDeviceConfig()
        if device is not None:
            path = device.pop("ConfigurationPath")
            poppedValue = device.pop(config)
            with open(path, 'w') as file:
                file.write(json.dumps(json.loads(json.dumps(device)), indent=4, sort_keys=True))
            print("Popped {0}".format(config, poppedValue))
            return True
    except Exception as e:
        print("Unable to save Config: " + str(e))
        exit()
    return None

def logProcess(processID):
    # Only Windows' System
    try:
        device = getDeviceConfig()
        #Other systems don't need to log
        if device["Device"] in ["MM", "Server"]:
            dir_path = os.path.dirname(os.path.realpath(__file__))
            backupPath = os.path.join(dir_path, "processes")
            if not os.path.exists(backupPath):
                os.mkdir(backupPath)
            file = open(os.path.join(backupPath, str(processID)), 'w+')
            print("ProcessID logged: " + str(processID) )
        return True
    except Exception as e:
        print(e)
        return False

def writeBackup():
    #shouldn't be used frequently
    dir_path = os.path.dirname(os.path.realpath(__file__))
    backupPath = os.path.join(dir_path, "BackhaulConfigurationBackup.conf")
    try:
        device = getDeviceConfig()
        path = device.pop("ConfigurationPath")
    except ValueError as e:
        print("The old device file is corrupted.")
        print("Not saving this to the backup config.")
        return False
    with open(backupPath, 'w') as file:
        file.write(json.dumps(json.loads(json.dumps(device)), indent=4, sort_keys=True))
    print("Backup File Saved!")
    return True

def readBackup():
    #file should only be touched at write and only read when the first json fails
    dir_path = os.path.dirname(os.path.realpath(__file__))
    backupPath = os.path.join(dir_path, "BackhaulConfigurationBackup.conf")
    
    if os.path.exists(backupPath):
        try:
            with open(backupPath, "r") as file:
                backupConfiguration = json.loads(file.read())
                print("Reading Backup Configuration for " + backupConfiguration["Device"])
        except ValueError as e:
            print("Backup file is corrupted.")
            return False
        newSave = os.path.join(dir_path, "BackhaulConfiguration.conf")
        with open(newSave, 'w') as file:
            file.write(json.dumps(json.loads(json.dumps(backupConfiguration)), indent=4, sort_keys=True))
        return True
    else:
        print("No backup file!")
        return False



if __name__ == '__main__':
    print(updatePulsarAddress())
